
function runUpscalingForDFN(gridSize, DFNname)
    clc;close all;
    %% LOAD MRST MODULES
    mrstModule add hfm ad-blackoil ad-core ad-props
    
    %% FRACTURE NETWORK STATISTICAL PARAMETERS
    physdim=[50 50 1]; % domain size
    
    %% DFN GENERATION
    tol=10^-5; % tolerance for comparison of doubles
    fracplanes=[]; % empty list of fracture objects

    % Define the folder where you want to save the results
    resultsFolder = strcat('Km1Intensities001/', DFNname,'-',num2str(gridSize));
    % Path to the location of input data (coordinate)
    inputFolderCoordinate = strcat('DFNs/', DFNname, '/fractureCoordinates/001fractureCoordinates.txt');
    % Read 2D fracture data
    fractureLines = readFractureData(inputFolderCoordinate);

    % Path to the location of input data (aperture)
    inputFolderaperture = strcat('DFNs/', DFNname, '/aperture/001aperture.txt');
    fractureApertures = readFractureAperture(inputFolderaperture);
    
    % Extrude these lines into 3D planes
    depth = physdim(3); % Depth of extrusion
    fracplanes = extrudeFracturePlanes(fractureLines,fractureApertures, depth);

    %% PLOT DFN
    colourchoice=['r','g','y'];
    figure;
    hold on;
    axis equal tight
    view(45,30);
    for i=1:3
        index = find(vertcat(fracplanes.SetID)==i);
        C=colourchoice(i);
        for j=index'
            X=fracplanes(j).points(:,1);
            Y=fracplanes(j).points(:,2);
            Z=fracplanes(j).points(:,3);
            fill3(X,Y,Z,C,'FaceAlpha',0.7); % Added 'FaceAlpha' to control the transparency of fracture planes
        end
    end
    xlabel('x','Interpreter','latex')
    ylabel('y','Interpreter','latex')
    zlabel('z','Interpreter','latex')
    
    %% Gridding and upscaling
    % Define grid sizes and permeability values for sensitivity analysis
    %gridSizes = [100];%, 100,200,400,800,1600
    perm=0.1;
    k_eq_values = zeros(1, 1); % Store k_eq for each combination
    passed=true;
    % Loop through each permeability 
    % Grid setup
    celldim = [gridSize gridSize 1];
    G1 = cartGrid(celldim, physdim); % Assume 1 as the physical z-dimension
    G1 = computeGeometry(G1);
    tic
    try
        km = perm * milli*darcy; % Convert permeability to Darcy units
        G1.rock = makeRock(G1,km, 0.25); % Use current permeability, assume porosity is constant
        
       %% EDFM PRE-PROCESSING
        tol=1e-6;
        [G1,fracplanes]=EDFMgrid(G1,fracplanes,'Tolerance',tol);
        G1=fracturematrixNNC3D(G1,tol);
        [G1,fracplanes]=fracturefractureNNCs3D(G1,fracplanes,tol);
        
        %% SETUP FLUID MODEL WITH WATER PROPERTIES
        pRef = 100*barsa;
        fluid = initSimpleADIFluid('phases','W',...
        'mu' , 1* centi*poise     , ...
        'rho', 1000* kilogram/meter^3, ...
        'pRef',    pRef, ...
        'c',       0/barsa);
        
        %% SETUP WATER MODEL
        gravity reset off
        model = WaterModel(G1, G1.rock, fluid);
        model.operators = setupEDFMOperatorsTPFA(G1, G1.rock, tol);
        model.stepFunctionIsLinear=true;
        
        %% INITIAL AND BOUNDARY CONDITIONS
        % Initial condition
        state  = initResSol(G1, pRef); 
        
        % Find fracture cell faces at domain boundary
        boundfaces=findfracboundaryfaces(G1,tol);
        
        % Set pressure differential on opposing boundaries in the x-direction
        deltaP = 50*barsa;
        bc=[];
        bc=pside(bc,G1.Matrix,'North',pRef);
        matwestfaces=bc.face;
        bc=pside(bc,G1.Matrix,'South',pRef + deltaP);
        bc=addBC(bc,boundfaces.North,'pressure',100*barsa);
        bc=addBC(bc,boundfaces.South,'pressure',150*barsa);
        bc.sat=ones(size(bc.face));
        
        %% SETUP SCHEDULE
        schedule = simpleSchedule(repmat(0.1, 10, 1),'bc',bc);
        
        %% SIMULATE
        [~, states,~] = simulateScheduleAD(state, model, schedule);
        westfaces=[matwestfaces;boundfaces.West'];
        waterflux=sum(abs(states{end}.flux(westfaces,1)));
        
        % Darcy's law
        k_eq=waterflux*fluid.muW(1)*physdim(1)/(physdim(3)*physdim(2)*deltaP);
        % Insert logic to calculate k_eq based on simulation results for the current grid size and permeability
        k_eq_values(1, 1) = convertTo(k_eq, milli*darcy); % Store k_eq result

    catch ME
        passed=false;
        % Log error message for debugging
        fprintf('Error occurred for Grid Size: %d, Permeability: %f mD. Error Message: %s\n', gridSize, perm, ME.message);
        % Set k_eq to NaN or some indicative value
        k_eq_values(1, 1) = NaN;
        % Continue to next iteration
    end
toc

    %% Define path for outputs
    % Check if the folder exists, if not, create it
    if ~exist(resultsFolder, 'dir')
        mkdir(resultsFolder);
    end

    % Change MATLAB's current directory to the results folder
    cd(resultsFolder);

    %%  Save k_eq_values to a text file
    dlmwrite('k_eq_1.txt', k_eq_values, 'delimiter', ',', 'precision', 9);
    %% PLOT RESULTS
    numTimesteps = numel(states); % Number of timesteps in the simulation
    if passed
        for step = 1:numTimesteps
            fprintf('Saving pressure profile for timestep %d\n', step);
            figure;
            % Plotting matrix grid cells
            plotCellData(G1, states{step}.pressure, 1:G1.Matrix.cells.num, ...
                'FaceAlpha', 0.5, 'EdgeAlpha', 0.1);
            % Plotting fracture cells if present
            plotCellData(G1, states{step}.pressure, G1.Matrix.cells.num + 1:G1.cells.num);
            view(30, 45);
            caxis([100 150] * barsa);
            colorbar('EastOutside');
            axis equal tight;
            box on;
            
            % Save the figure
            saveas(gcf, sprintf('pressureGradient_timestep%d.png', step)); % Save as PNG
            savefig(sprintf('pressureGradient_timestep%d.fig', step)); % Save as MATLAB FIG format
        end
    end
    
    %% PLOT DFN
    colourchoice=['r','g','y'];
    figure;
    hold on;
    axis equal tight
    view(45,30);
    for i=1:3
        index = find(vertcat(fracplanes.SetID)==i);
        C=colourchoice(i);
        for j=index'
            X=fracplanes(j).points(:,1);
            Y=fracplanes(j).points(:,2);
            Z=fracplanes(j).points(:,3);
            fill3(X,Y,Z,C,'FaceAlpha',0.7); % Added 'FaceAlpha' to control the transparency of fracture planes
        end
    end
    xlabel('x','Interpreter','latex')
    ylabel('y','Interpreter','latex')
    zlabel('z','Interpreter','latex')
    
    % Now save the figure
    saveas(gcf, 'DFN3D.png');
    savefig('DFN3D.fig'); % For MATLAB FIG format


cd ../../